
;MAXFLASH.ASM
;Firmware-based Flash utility for use with Adapt812MX2 Memory Expansion and Adapt812MAX
;works with AMD AM29F400
;Instructions:
;============
;Load this code (MAXFLASH.S19) into 68HC812A4 on-chip EEPROM
;Then reset with MODA & MODB jumpered for single-chip mode (ie. both low)
;This code switches on external wide bus in page mode and accesses external Flash
;connected to CSP0*.  To use pages, user must embed page number in s-record
;prior to the s-record segment to be loaded into a given page
;This can be done with assembler directives.  For example, to place a group
;of subroutines in PPAGE3 program space, preface the group of routines
;as follows:
;	org	PPAGE
;	fcb	$03
;	org	$8000
;put subroutines here...
;
;The user should always preface the vector block with:
;
;	org	PPAGE
;	fcb	$1f	;512K shown (use $3f for 1M Flash)
;	org	$bf80	;start of vector table
;put vectors here...
;
;When Flash has been programmed, jumper MODA & MODB for Normal Expanded Narrow
;mode (ie. MODA high) then reset.  The internal EEPROM will move to $1000, and
;the user code in Flash will now execute.
;==============================================================================
;
*******************************************************************************
*REVISION HISTORY:
*
*DATE          REV. NO.		DESCRIPTION
*Jun 28,2000	0.00		-released
*******************************************************************************
;
; ----------------------------
; ADAPT812MAX1 Flash Utilities
; ----------------------------
;REGS.INC is an external file containing the 68HC812A4 register definitions
;
#include REGS.INC

*  Operational Parameters

_100MS          equ     25
_250MS          equ     61
_500MS          equ     125
_1SECOND        equ     250
_2SECONDS       equ     500
_3SECONDS       equ     750
_5SECONDS       equ     1250
_10SECONDS      equ     2500
_25SECONDS      equ     6250
_60SECONDS      equ     15000
_2MIN           equ     30000

RAM     equ     $0800           ;68HC812A4 internal RAM
STACK   equ     $0c00           ;Stack at top of internal ram
rbase   equ     $0000           ;68HC812A4 register block
EEPROM  equ     $f000           ;68HC812A4 internal EEPROM in SCM
SRAM	equ	$7000		;start of external SRAM, when enabled
DWIN	equ	$7000		;start of data window
ENDDWIN	equ	$7fff		;end of data window
FLASH	equ	$8000		;start of Flash
PWIN	equ	$8000		;start of program window
ENDPWIN	equ	$bfff		;end of program window
LDP128	equ	$1f		;last 4K page of 128K data space
LDP512	equ	$7f		;last 4K page of 512K data space
LPP128	equ	$07		;last 16K page of 128K program space
LPP512  equ     $1f             ;last 16K page of 512K program space
ex      equ     $8002

* Operational Constants

true            equ     $ff
false           equ     $00
CR              equ     $D
LF              equ     $A
SPACE           equ     $20

* Masks

;SCI Variables
scimask         equ     %00101100 ;RIE - SCI Interrupt enable
                                  ;RE - Receiver Enable
RDRFflag        equ     %00100000 ;RDRF - Receive Data Register Full flag
TDREflag        equ     %10000000 ;TDRE - Transmit Data Register Empty flag

;Baud rate definitions
;MCLK=8MHz
BAUD110         equ     4545    ;(baud) 110 baud with 16 Mhz crystal
BAUD300         equ     1667    ;(baud) 300 baud with 16 Mhz crystal
BAUD600         equ     833     ;(baud) 600 baud with 16 Mhz crystal
BAUD1200        equ     417     ;(baud) 1200 baud with 16 Mhz crystal
BAUD2400        equ     208     ;(baud) 2400 baud with 16 Mhz crystal
BAUD4800        equ     104     ;(baud) 4800 baud with 16 Mhz crystal
BAUD9600        equ     52      ;(baud) 9600 baud with 16 Mhz crystal
BAUD14400       equ     35      ;(baud) 14400 baud with 16 Mhz crystal
BAUD19200       equ     26      ;(baud) 19200 baud with 16 Mhz crystal
BAUD38400       equ     13      ;(baud) 38400 baud with 16 Mhz crystal

        org     RAM

* System Variables

BYTE            rmb     2       ;used for data polling
CNT1		rmb	2
CNT2		rmb	2
TEMP1		rmb	1
TEMP2		rmb	1
TEMP3           rmb     2
S_ADR		rmb	2

CURPAGE		rmb	1	;current page counter
MEM_SIZE	rmb	2	;memory size (128 or 512)
LAST_PPAGE	rmb	1	;last page of Flash
LAST_DPAGE	rmb	1	;last page of RAM

H               rmb     1       ;used in binary to decimal conversion
TO              rmb     1
byte_counter    rmb     1


* Boolean Variables

ERROR		rmb	1	;Flash operation status
WRITE_MODE	rmb	1	;write or verify
BANKING		rmb	1	;banking or non-banking mode
address_flag    rmb     1

***************************** Program *******************************

        org     EEPROM

Init:	                        ;This is where the RESET vector points to
;Initialize Stack
        lds     #STACK          ;initialize stack pointer

;Set up external memory control registers
        movb    #$0c,PEAR       ;enable read/write signal and lstre
        movb    #$f0,MODE       ;normal exp. wide
	movb	#$c0,WINDEF	;enable data & program windows
        movb    #$0F,MXAR       ;enable A16, A17, A18, A19
	movb	#$30,CSCTL0	;enable CSD & CSP0
	movb	#$10,CSCTL1	;CSD covers lower half of memory map
	movb	#$f1,INITEE	;make sure EEPROM is at top

;Initialize COP
	clr	COPCTL			;turn COP off

;Initialize Serial Communication Interface
        movb    #$0c,SC0CR2		;enable SCI 0 rx & tx
        movb    #0,SC0CR1		;  for polled operation
        movw    #BAUD9600,SC0BDH       ;Set baud rate

	movb	#$ff,DDRT	;make port t outputs

        clr     address_flag
        ldx     #512            ;default memory size is 128K
	stx	MEM_SIZE

        movb    #LPP512,LAST_PPAGE
        movb    #LDP512,LAST_DPAGE
	movb	#true,BANKING	;default to banked mode

	jsr	ShowMenu	;display menu in terminal window

main
	jsr     ProcessCommand  ;this is the program loop
	jsr	ShowMenu	;display menu in terminal window
        bra     main


***********************************************************************
*Procedure Definitions:
***********************************************************************

ShowMenu:
        jsr     PutDblLine
        ldx     #M_UNDLN
        jsr     SendString
        jsr     PutDblLine
        ldx     #M_HEADER1
        jsr     SendString
        jsr     PutNewLine
        ldx     #M_UNDLN
        jsr     SendString
        jsr     PutNewLine
	ldx	MEM_SIZE
	cpx	#512
	bne	SM128
        ldx     #M_MEM512
	bra	SM_MEM
SM128:
	ldx	#M_MEM128
SM_MEM:
        jsr     SendString
	tst	BANKING
	bne	B_ON
	ldx	#M_BANKING_OFF
	bra	SM_BANKING
B_ON:
	ldx	#M_BANKING_ON
SM_BANKING:
	jsr	SendString
        jsr     PutDblLine

;        ldx     #M_1MEM
;        jsr     SendString
;        jsr     PutNewLine
;        ldx     #M_5MEM
;        jsr     SendString
;        jsr     PutNewLine
        ldx     #M_BANKING
        jsr     SendString
        jsr     PutNewLine
        ldx     #M_LOAD
        jsr     SendString
        jsr     PutNewLine
;        ldx     #M_RAMTEST
;        jsr     SendString
;        jsr     PutNewLine
        ldx     #M_ERASE
        jsr     SendString
        jsr     PutNewLine
        ldx     #M_PROGRAM
        jsr     SendString
        jsr     PutNewLine

        ldx     #M_VERIFY
        jsr     SendString
        jsr     PutNewLine

;        ldx     #M_CODE
;        jsr     SendString
;        jsr     PutNewLine

        ldx     #M_PROMPT
        jsr     SendString
        rts


ProcessCommand:
	jsr	GetChar
	jsr	SendByte	;echo the character
	oraa	#$20		;convert to lower case, if required

        cmpa    #'1'		;set memory size (128K)
;        lbeq    PCM1

        cmpa    #'5'		;set memory size (512K)
;        lbeq    PCM5

        cmpa    #'n'		;set banking/non-banking mode
        lbeq    PCMN

        cmpa    #'l'		;load s-record into RAM
        lbeq    PCL

        cmpa    #'r'		;do RAM test
        lbeq    PCR

        cmpa    #'e'		;erase Flash
        lbeq    PCE

        cmpa    #'p'		;program s-record into Flash
        lbeq    PCP

        cmpa    #'m'            ;read/set time
        lbeq    prg_location

        cmpa    #'v'		;verify s-record
        lbeq    PCV

        cmpa    #'c'            ;product code
        lbeq    PCC

	jmp	PCX		;  and exit


PCM1:
	ldy	#128
	ldx	#M_MEM128
	movb	#LPP128,LAST_PPAGE
	bra	PCMX
PCM5:
	ldy	#512
	movb	#LPP512,LAST_PPAGE
	ldx	#M_MEM512
PCMX:
	sty	MEM_SIZE
	jsr	PutNewLine
	jsr	SendString	;display new memory-size message
	jsr	PutNewLine
        jmp     PCX

PCMN:
	jsr	PutNewLine
	com	BANKING		;toggle bank/non-banked mode
	ldx	MEM_SIZE	;display new memory configuration message
	cpx	#512
	bne	PSM128
        ldx     #M_MEM512
	bra	PSM_MEM
PSM128:
	ldx	#M_MEM128
PSM_MEM:
        jsr     SendString
	tst	BANKING
	bne	PB_ON
	ldx	#M_BANKING_OFF
	bra	PSM_BANKING
PB_ON:
	ldx	#M_BANKING_ON
PSM_BANKING:
	jsr	SendString
        jsr     PutDblLine
        jmp     PCX

PCC:
        clr     PPAGE

        ldaa    #$F0		;need to reset flash
        std     $9000

        movw    #$AAAA,$8AAA       ;need offset of $8000 to generate CSP0*
        movw    #$5555,$8554       ;Flash needs address $2aaa (in first page)
        movw    #$9090,$8AAA 
		
        ldd     $8000		;read man. id.
        pshb
        psha
        pula
        jsr     SendASCIIHex
        pula
        jsr     SendASCIIHex
	
	jsr     PutNewLine

        ldd     $8001		;read device id.
        pshb
        psha
        pula
        jsr     SendASCIIHex
        pula
        jsr     SendASCIIHex
     
	
	jsr     PutNewLine
        
	ldd     $8002		
        pshb
        psha
        pula
        jsr     SendASCIIHex
        pula
        jsr     SendASCIIHex
    

        ldaa    #$F0
        std     $9000

        jmp     PCX

PCL:
	jsr     PutNewLine
	ldx	#M_PROMPT1
	jsr	SendString
	jsr     PutNewLine
	movb	#true,WRITE_MODE	;select non-verify mode
        jsr	ProcessSRecord
	tst	ERROR
	bne	PCL_Fail
	ldx	#M_LOAD_OK
	bra	PCL_SS
PCL_Fail:
	ldx	#M_LOAD_FAIL
PCL_SS:
	jsr	SendString
        jmp     PCX

PCR:
	jsr	PutNewLine
	ldx	#M_RAMTEST
	jsr	SendString
	jsr	PutNewLine
        jsr	TestRAM		;test RAM
        jmp     PCX

PCE:
	jsr	PutNewLine
	ldx	#M_ERASING
	jsr	SendString
	jsr	PutNewLine
        jsr	EraseFlash	;erase Flash
        jmp     PCX
PCP:
	jsr     PutNewLine
	ldx	#M_PROMPT1
	jsr	SendString
	jsr     PutNewLine
	movb	#true,WRITE_MODE	;select non-verify mode
        jsr	ProcessSRecord
	tst	ERROR
	bne	PCP_Fail
	ldx	#M_PROG_OK
	bra	PCP_SS
PCP_Fail:
	ldx	#M_PROG_FAIL
PCP_SS:
	jsr	SendString
        bra     PCX

PCV:
	jsr     PutNewLine
	clr	WRITE_MODE	;select verify mode
	ldx	#M_SEND_S19
	jsr	SendString
	jsr     PutNewLine
        jsr	ProcessSRecord
	jsr     PutNewLine
	tst	ERROR
	bne	PCV_Fail
	ldx	#M_VERIFY_OK
	bra	PCV_SS
PCV_Fail:
	ldx	#M_VERIFY_FAIL
PCV_SS:
	jsr	SendString
        bra     PCX

PCX:
        rts


prg_location
        ldd     #$2020
        ldy     #$8000

        jsr     WriteFlash 

        ldy     #$8000
        ldaa    1,y
        jsr     SendASCIIHex

        jmp     main

;-----------------------------------------------------------------------------
EraseFlash:
        movb    #$ff,PORTT      ;LED on
	clr	ERROR

        clr     PPAGE
        ldaa    #$F0
        std     $9000

        movw    #$AAAA,$8AAA       ;need offset of $8000 to generate CSP0*
        movw    #$5555,$8554       ;Flash needs address $2aaa (in first page)
        movw    #$8080,$8AAA       ;3
        movw    #$AAAA,$8AAA       ;4
        movw    #$5555,$8554       ;5
        movw    #$1010,$8AAA       ;6

EF1
        ldd     FLASH
        rolb
        bcc     EF2
        bra     EF_OK

EF2
        rolb
        rolb
        bcc     EF1

        ldd     FLASH
        rolb
        bcs     EF_OK

	ldx	#M_ERASE_FAIL	;  send FAIL message
	bra	EF_SS

EF_OK:
	ldx	#M_ERASE_OK	;  send OK message
EF_SS:
	clr	PORTT		;LED off

        jsr     SendString
        jsr     PutNewLine

	rts


TestRAM:
	ldx	#M_NOT		;not implemented
	jsr	SendString
	jsr	PutNewLine
	rts
************************************
*  RAMtest not yet implemented

PCT2:				;write command sequence
	clr	ERROR
	bne	PCT_Fail
PCT_OK:
	iny
	cpy	#$c000		;until end of page
	bne	PCT2
Done:
	ldx	#M_TEST_OK	;  send OK message
	bra	PCT_SS
PCT_Fail:
	ldx	#M_TEST_FAIL	;  send FAIL message
PCT_SS:
	jsr	SendString
	jsr     PutNewLine
	rts
************************************


ProcessSRecord 
	clr	ERROR		;no errors yet

	tst	BANKING
	bne	PS_BANKED_MODE
	movb	LAST_PPAGE,PPAGE	;  start at second-last page if non-banked mode
	dec	PPAGE
M0:

PS_BANKED_MODE 
	jsr	GetChar		;look for S record
	cmpa	#'S
	bne	M0

	jsr	GetChar
	cmpa	#'1		;if not S1 record then
	beq	Load1

	cmpa	#'9		;if S9 record then
	bne	M0
	
	jsr	GetByte		;get byte count
       suba    #2
        staa    byte_counter
	jsr	GetAddress	;get execution address

Load9 
	jsr	GetByte		;get checksum byte (and ignore it)
        dec     byte_counter
	bne	Load9
	lbra	PSX

Load1 
	jsr	GetByte		;S1 record, so get byte count
        suba    #3
        staa    byte_counter

	jsr	GetAddress	;get address

        sty	S_ADR

	pshy
        puld
        lsrd                    ;check for odd address
        bcs     set_odd_address 
        bra     set_normal_address

set_odd_address 
	lsld
        movb    #1,address_flag
        bra     odd10

set_normal_address
	lsld
        clr     address_flag

odd10 
	cpy	#$c000		;if address is in last page
	blo	LO_HALF		;  subtract $4000 and set PPAGE to LAST_PAGE
	pshy
	puld
	subd	#$4000
	pshd
	puly
	movb	LAST_PPAGE,PPAGE	

LO_HALF                         ;Reg Y contains the address
        brclr   address_flag,%01,even_address
        dey
        ldaa    0,y
        psha
        bra     odd_address

even_address
	
	jsr	GetByte		;  get a byte (in A)
        psha                    ;Save high byte
        dec     byte_counter
	bne	odd_address
	cpy	#FLASH		;    if address below Flash then
	bhs	L2C
	pula
        staa    0,y             ;      do normal write (it's RAM or register)
	lbra	M0
L2C:	
	ldab	#$ff
	pula
	bra	L1C
		
odd_address
        jsr     GetByte
        dec     byte_counter
	
        tab                     ;transfer to lower byte
        pula                    ;Get high byte

        cpy	#FLASH		;    if address below Flash then
	bhs	L1C
        std     0,y             ;      do normal write (it's RAM or register)
	bra	VF

L1C
	tst	WRITE_MODE	;    else if not Verify then
	beq	VF
	jsr	WriteFlash	;     do Flash write
        ldd     BYTE

VF
        cpd     0,y             ;if byte doesn't verify then
	beq	VerfOK

	movb	#$40,PORTT	;turn on PT6 LED
        movb    #$ff,ERROR      ;set error flag

VerfOK
        iny                     ;The address should be even at this point
        iny

        ldab    byte_counter
        bne     even_address
        lbra    M0              ;get next S-record

PSX
	rts

;-----------------------------------------------------------------------------
WriteFlash 

	movb	PPAGE,CURPAGE	;preserve PPAGE value
        clr     PPAGE           ;Flash needs to see address of $8554,
				;which is in second 16K page of Flash
        movw    #$AAAA,$8AAA    ;need offset of $8000 to generate CSP0*
        movw    #$5555,$8554    ;
        movw    #$A0A0,$8AAA    ;

	movb	CURPAGE,PPAGE	;restore PPAGE value
        std     BYTE     
        movw    BYTE,0,y        ;write Flash


DataPoll         
        ldd     BYTE             
	staa	TEMP1
	stab	TEMP2
DP1
	ldd	0,y		;get value being programmed
	eorb	TEMP2
	bpl	DPP	;until DQ7 matches data bit7 or
CheckDQ5 
        rolb
        rolb
        rolb
        bcc     DataPoll

	movw	#$1000,CNT2
	movw	#$ffff,CNT1
DPP	
	ldd	0,y		;verify data
	cpd	BYTE
	beq	DPX
	dec	CNT1
	bne	DPP
	dec	CNT2
	beq	Err
	ldd	BYTE
	bra	WriteFlash

Err:
	movb	#$40,PORTT	;turn on PT6 LED
	movb    #$ff,ERROR      ;set error flag

DPX 
	rts

;-----------------------------------------------------------------------------
HexByte2Dec             ;Supply hex value to be converted in B register
        clra		;numerator in D, denominator in X
        ldx     #100    ;remainder in D, quotient in X
        idiv		;B is lo-order byte
        xgdx
        stab    H       ;8 bit quotient in B
        xgdx
        ldx     #10     ;H=0X  TO=packed BCD
        idiv
        stab    TO
        xgdx
        aslb
        aslb
        aslb
        aslb
        orab    TO
        stab    TO
        rts


GetChar:

*  Fetches one character from SCI0, 
*  and returns it in register A.  

GC1:
	ldaa	SC0SR1		;get sci flags
	anda	#RDRFflag	;mask off irrelevant bits
	beq	GC1		;loop until character found
	ldaa	SC0DRL		;get character
	rts


GetByte:

*  Converts 2 hex characters to a single byte value, 
*  and returns it in register A.

GB1:
	jsr	GetChar
	jsr	Hex2Bin
	lsla
	lsla
	lsla
	lsla
	staa	TEMP1
	jsr	GetChar
	jsr	Hex2Bin
	oraa	TEMP1
	rts


Hex2Bin:

*  Converts an ASCII hex character in register A
*  to a binary nibble and returns it in register A.

	cmpa	#'9
	bls	HEX
	adda	#9
HEX:	anda	#$f
	rts

GetAddress:

* Extracts load address and returns it in register Y.

	pshb			;preserve byte count
	jsr	GetByte
	staa	TEMP2
	jsr	GetByte
	tab
	ldaa	TEMP2
	xgdy
	pulb
	rts	


SendDecByte

* value is in B

        clra
        jsr     HexByte2Dec
        ldaa    H
        adda    #'0'
        jsr     SendByte
        ldaa    TO
        jsr     SendASCIIHex
        rts

SendASCIIHex
*       value in A

        psha
        anda    #$F0
        lsra
        lsra
        lsra
        lsra
        cmpa    #$A
        blt     SAH1
        suba    #$A
        adda    #'A'
        bra     SAH2
SAH1
        adda    #'0'
SAH2
        jsr     SendByte
        pula
        psha
        anda    #$0F
        cmpa    #$A
        blt     SAH3
        suba    #$A
        adda    #'A'
        bra     SAH4
SAH3
        adda    #'0'
SAH4
        jsr     SendByte
        pula
        rts


SendByte:

*  Transmits a byte, contained in register A, to the serial device via the SCI.

SB1
        ldab    SC0SR1  ;wait for transmit data register empty (TDRE)
        bpl     SB1
        staa    SC0DRL  ;send byte
        rts

PutNewLine
        ldaa    #CR
        jsr     SendByte
        ldaa    #LF
        jsr     SendByte
        rts

PutDblLine
        ldaa    #CR
        jsr     SendByte
        ldaa    #LF
        jsr     SendByte
        ldaa    #LF
        jsr     SendByte
        rts

SendString:
*       x contains starting address of string
RV1
        ldaa    0,X             ;$FF denotes end of string
        cmpa    #$ff
        beq     RVX
        jsr     SendByte
        inx
        bra     RV1
RVX
        rts

*******************************************************************************
* Messages

*******************************************************************************
* Messages
M_1MEM		fcc     '1 => Memory devices are 128K'
                fcb     $ff
M_5MEM		fcc     '5 => Memory devices are 512K'
                fcb     $ff
M_BANKING	fcc     'N => Toggle Banked/Non-banked Memory Mode'
                fcb     $ff
M_LOAD		fcc     'L => Load s-record into RAM'
                fcb     $ff
M_RAMTEST	fcc     'R => Perform RAM test'
                fcb     $ff
M_ERASE         fcc     'E => Erase entire Flash chip'
                fcb     $ff
M_PROGRAM       fcc     'P => Program s-record into Flash'
                fcb     $ff

M_VERIFY        fcc     'V => Verify memory contents matches s-record'
                fcb     $ff

M_CODE          fcc     'C => Product number code  '
                fcb     $ff

M_PROMPT        fcc     '?'
                fcb     $ff
M_PROMPT1	fcc     'Send s-record to be programmed'
                fcb     $ff
M_HEADER1       fcc     ' ADAPT812MAX & MX2 FLASH/RTC UTILITY VERSION 0'
                fcb     $ff
M_UNDLN         fcc     '_________________________________________________'
                fcb     $ff
M_ERASE_OK	fcc     'Flash erased OK'
                fcb     $ff
M_ERASE_FAIL    fcc     'Could not erase Flash'
                fcb     $ff
M_ERASING	fcc     'erasing Flash now...'
                fcb     $ff
M_TESTING	fcc     'testing RAM now...'
                fcb     $ff
M_LOADING	fcc     'loading RAM now...'
                fcb     $ff
M_VERIFY_OK     fcc     's-record verified OK'
                fcb     $ff
M_VERIFY_FAIL   fcc     's-record verification failed'
                fcb     $ff
M_PROG_OK       fcc     'Flash programmed OK'
                fcb     $ff
M_PROG_FAIL     fcc     'Flash programming failed'
                fcb     $ff
M_LOAD_OK       fcc     'RAM loaded OK'
                fcb     $ff
M_LOAD_FAIL     fcc     'RAM load failed'
                fcb     $ff
M_TEST_OK       fcc     'RAM test successful'
                fcb     $ff
M_TEST_FAIL	fcc     'RAM test failed'
                fcb     $ff
M_MEM128	fcc     '128K memory devices'
                fcb     $ff
M_MEM512	fcc     '512K memory devices'
                fcb     $ff
M_BANKING_OFF	fcc     '  Non-banked Mode'
                fcb     $ff
M_BANKING_ON	fcc     '  Banked Mode'
                fcb     $ff
M_SEND_S19	fcc	'Send s-record to be verified'
		fcb	$ff
M_NOT		fcc     'Not implemented'
		fcb	$ff

;MC68HC812A4 VECTOR INTERRUPTS

	org	$FFCE
VECTORS:
        FDB     Init           ;KEY WAKEUP H
        FDB     Init           ;KEY WAKEUP J
        FDB     Init           ;ANALOG TO DIGITAL
        FDB     Init           ;SERIAL COMMUNICATION 1
        FDB     Init	       ;SERIAL COMMUNICATION 0
        FDB     Init           ;SPI SERIAL TRANSFER COMPLETE
        FDB     Init           ;PULSE ACCUMULATOR INPUT EDGE
        FDB     Init           ;PULSE ACCUMULATOR OVERFLOW
        FDB     Init           ;TIMER OVERFLOW
        FDB     Init           ;TIMER CHANNEL 7
        FDB     Init           ;TIMER CHANNEL 6
        FDB     Init           ;TIMER CHANNEL 5
        FDB     Init           ;TIMER CHANNEL 4
        FDB     Init           ;TIMER CHANNEL 3
        FDB     Init           ;TIMER CHANNEL 2
        FDB     Init           ;TIMER CHANNEL 1
        FDB     Init           ;TIMER CHANNEL 0
        FDB     Init	       ;REAL TIME INTERRUPT
        FDB     Init           ;IRQ OR KEY WAKE UP D
        FDB     Init           ;XIRQ
        FDB     Init           ;SWI
        FDB     Init           ;RESERVED
        FDB     Init           ;COP FAILURE RESET
        FDB     Init           ;COP CLOCK MONITOR FAIL RESET
        FDB     Init           ;RESET

